--[[

Modified by Yoshi for 600 mm gauge.

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
NOTICE:
If you change something for your own purposes you have to change the filename to avoid
incompatibilities with existing modifications and game crashes.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

Copyright (c) 2016 "Tom" from www.transportfever.net
(User page: www.transportfever.net/index.php/User/20244-Tom/)

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and 
associated documentation files (the "Software"), to deal in the Software without restriction, 
including the right to distribute and without limitation the rights to use, copy and/or modify 
the Software, and to permit persons to whom the Software is furnished to do so, subject to the 
following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial 
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT 
NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
--]]

local vec3 = require "vec3"
local transf = require "transf"
local edgeutil = require "tom_edgeutil_v1_1"

local StationBuilder = { }

local PLANUM_AREA = 5           -- size of area added when base heigth of station <> 0 to avoid terrain collisions when snapping tracks
local TRACK_EXTENSION = 2       -- length of track added on both sides of the station

-- Replaces the epoch placeholder "${epoch}" in the given filename with the value in config.epoch.
-- If config.epoch doesn't exist, the filename is returned unmodified.
local function epochFilename(config, filename)
    if config.epoch then
        filename = filename % { epoch = config.epoch }
    end
    return filename
end

----------------------------------------------------------------------------
-- public functions for UI-Parameters

function StationBuilder.platformLengthUiParam(config, defaultIndex)
	local values = { }
	local segmentLength = config.segmentLength
	local items = config.availableSizes
	local start = config.minAvailableSizeIndex or 1
	for i = start, #items do
        values[#values + 1] = _("${value}m") % { value = items[i] * segmentLength }
    end
	return {
		key = "tom_sizeIndex",
		name = _("Platform length"),
		uiType = "SLIDER",
		values = values,
		defaultIndex = defaultIndex or 1
	}
end

function StationBuilder.numberOfTracksUiParam(config, defaultIndex)
	local values = { }
	local items = config.availableNumberOfTracks
	for i = 1, #items do
        values[#values + 1] = _("${value}") % { value = items[i] }
    end
	return {
		key = "tom_numTracksIndex",
		name = _("Number of tracks"),
		uiType = "SLIDER",
		values = values,
		defaultIndex = defaultIndex or 0
	}
end

function StationBuilder.platformRoofingUiParam(config, defaultIndex)
    return {
		key = "tom_platformRoofing",
		name = _("Platform roofing"),
		uiType = "CHECKBOX",
		values = { _("No"), _("Yes") },
	    defaultIndex = defaultIndex or 0
    }
end

function StationBuilder.trackTypeUiParam(config)
	return {
		key = "tom_trackType",
		name = _("600mm_depot_track_type_name"),
		uiType = "ICON_BUTTON",
		values = { 
			"ui/construction/params/tracks/600mm_holz_08.tga",
			"ui/construction/params/tracks/600mm_holz_12_gras.tga",
			"ui/construction/params/tracks/600mm_holz_12_schotter.tga",
			"ui/construction/params/tracks/600mm_holz_53_schotter.tga",
			"ui/construction/params/tracks/600mm_stahl_08.tga",
			"ui/construction/params/tracks/600mm_stahl_12_gras.tga",
			"ui/construction/params/tracks/600mm_stahl_12_schotter.tga",
			"ui/construction/params/tracks/600mm_stahl_53_schotter.tga",
		},
		defaultIndex = 2,
	}
end

function StationBuilder.trackCatenaryUiParam(config, defaultIndex)
	return {
		key = "catenary",
		name = _("Catenary"),
		values = { _("No"), _("Yes") },
		uiType = "CHECKBOX",
		defaultIndex = 0,
		yearFrom = 1910,
		yearTo = 0,
		defaultIndex = defaultIndex or 1		
	}
end

----------------------------------------------------------------------------
-- public function to initialize configuration values from UI-Parameters

function StationBuilder.initializeConfigParams(params, config)
    -- the UI doesn't adjust the indexes if we have fewer entries than an other station.
    -- So we take our maximums if the index is out of range    

    local index = (params.tom_numTracksIndex or 0) + 1
    index = index > #config.availableNumberOfTracks and #config.availableNumberOfTracks or index
    config.numberOfTracks = config.availableNumberOfTracks[index]    

    index = (params.tom_sizeIndex or 0) + (config.minAvailableSizeIndex or 1)
    index = index > #config.availableSizes and #config.availableSizes or index
    config.sizeIndex = index
    config.numberOfSegments = config.availableSizes[index]   
    
    index = (params.tom_trackType or 0) + 1
    index = index > #config.trackTypes and #config.trackTypes or index
    config.trackType = config.trackTypes[index]
    config.deadEndTrackTypes = config.deadEndTrackTypes or config.trackTypes
    config.deadEndTrackType = config.deadEndTrackTypes[index] 
    
	local trackLayout = config.trackLayouts[1]
	
    config.platformRoofing = params.tom_platformRoofing ~= 0
    config.catenary = params.catenary ~= 0

    config.trackDistance = params.state.track.trackDistance 
end

-- Adds a ground face to the station
--
-- Parameters:
--      config:     struct, required        - the configuration of the station
--      result:     struct, required        - the blueprint of the station for TpF
--      groundFace: struct, required        - the parts of the building that should be built
--
-- parameter "groundFace" has the following structur: 
--      left:       numeric, required       - x position of the left corner  
--      top:        numeric, required       - y position of the left corner
--      length:     numeric, required       - the length of the rectangle on the x-axis
--      width:      numeric, required       - the width of the rectangle on the y-axis
--      height:     numeric, optional,      - the height position of the rectangle
--                  default = config.baseHeight
function StationBuilder.addGroundFace(config, result, groundFace)
    local z = groundFace.height or config.baseHeight
    local face = {
        { groundFace.left,                    groundFace.top,                     z }, 
        { groundFace.left + groundFace.width, groundFace.top,                     z },
        { groundFace.left + groundFace.width, groundFace.top + groundFace.length, z }, 
        { groundFace.left,                    groundFace.top + groundFace.length, z } 
    }
    --result.terrainAlignmentLists[1].faces[#result.terrainAlignmentLists[1].faces + 1] = face
    result.terrainAlignmentLists[#result.terrainAlignmentLists + 1] = { type = "EQUAL", faces = { face } } 
    if groundFace.fill then
        result.groundFaces[#result.groundFaces + 1] = { face = face, modes = { { type = "FILL", key = "industry_concrete_01.lua" } } }
        result.groundFaces[#result.groundFaces + 1] = { face = face, modes = { { type = "STROKE_OUTER", key = "building_paving.lua" } } }
    end
end

-- Only for convenience. Adds an array of ground faces to the station
function StationBuilder.addGroundFaces(config, result, groundFaces)
    for i = 1, #groundFaces do        
        StationBuilder.addGroundFace(config, result, groundFaces[i])
    end
end

-- Adds a building to the station
--
-- Parameters:
--      config:     struct, required        - the configuration of the station
--      result:     struct, required        - the blueprint of the station for TpF
--      model:      struct, required        - the parts of the building that should be built
--               
-- parameter "model" has the following structure: 
--      building:       string, required                    - filename of the model
--      x:              numeric, optional, default = 0      - x position (coordinate system of the station)
--      y:              numeric, optional, default = 0      - the y position
--      height:         numeric, optional, 
--                      default = config.baseHeight         - the z position
--      rotateZ:        numeric, optional, default = 0      - rotation z axis, overrides global modelRotation in config
--      path:           string, optional                    - filepath relative to "res/models/model", overrides global modelPath in config
--      assets:         string, optional                    - filename of the model with assets
--      groundFace:     struct, optional                    - allign terrain area (see StationBuilder.addGroundFace)
--      groundFaces:    table, optional                     - or an array of ground faces
function StationBuilder.addBuilding(config, result, model)
    local path = model.path or config.modelPath
    local rotateZ = config.modelRotation or 0
    rotateZ = (model.rotateZ or rotateZ)
    local x = model.x or 0
    local y = model.y or 0
    local z = model.height or config.baseHeight

    result.models[#result.models + 1] = {
        id = epochFilename(config, path .. model.building),
        transf = transf.rotZYXTransl(transf.degToRad(rotateZ, 0, 0), vec3.new(x, y, z))
    }
       
    if (model.assets) then
        result.models[#result.models + 1] = {
            id = epochFilename(config, path .. model.assets),
            transf = transf.rotZYXTransl(transf.degToRad(rotateZ, 0, 0), vec3.new(x, y, z))
        }
    end

    if model.groundFace then        
        StationBuilder.addGroundFace(config, result, model.groundFace)
    end
    if model.groundFaces then
        StationBuilder.addGroundFaces(config, result, model.groundFaces)
    end
end

----------------------------------------------------------------------------
-- private things...

-- Since all types of edges (tracks, streets) are included in the build order in
-- result.edgeLists, we need to know the current number of entries in this table
-- when we add a track and want to specify the correct 'vehicleNodeOverride' node.
-- This node determines the position where the train stops, e.g. in the middle
-- of the platform or before the bumper of a dead-end track.
local function getTotalEdgeCount(result)
    local count = 0
    local edgeLists = result.edgeLists
    for i = 1, #edgeLists do
        count = count + #edgeLists[i].edges
    end
    return count
end

local function buildPlatform(config, result, trackDefinition, platformLayout)
    local height = config.baseHeight
    local sizeIndex = trackDefinition.sizeIndex
    local segments = platformLayout.segments
    local segmentLength = config.segmentLength
    local hasEndparts = (config.lengthEndSegment or 0) ~= 0

    local xOffset = trackDefinition.x + platformLayout.width / 2
    xOffset = xOffset + (platformLayout.moveX or 0)

    local length = config.availableSizes[sizeIndex] * segmentLength
    local yOffset = -length / 2 + segmentLength / 2
    
    local terminals = platformLayout.terminals or 2
    local terminals1 = { }
    local terminals2 = { }

    local reverse = platformLayout.reverse
    local start = reverse and #segments or 1
    local stop = reverse and 1 or #segments
    local step = reverse and -1 or 1
  
    for i = start, stop, step do	
        local model = segments[i]
              
        -- build the part of the platform only if it fits to the size 
        if model.sizeFrom <= sizeIndex and (not model.sizeTo or sizeIndex <= model.sizeTo) then

            -- If no platform is specified, then it is a dummy segment of a transit track
            if model.platform then
                if config.debug then 
                    print("            building platform segment: " .. model.platform)      
                end

                local path = model.path or config.modelPath
                local x = xOffset + (model.moveX or 0)
                local y = yOffset + (model.moveY or 0)
                local z = height + (model.moveZ or 0)
                local rotateZ = config.modelRotation or 0
                rotateZ = (model.rotateZ or rotateZ)
                rotateZ = rotateZ * step
                rotateZ = (reverse) and rotateZ + 180 or rotateZ
               
                local segmentTerminals = model.terminals or terminals
                if segmentTerminals >= 1 then
                    terminals1[#terminals1 + 1] = { #result.models, model.swapTerminals and 1 or 0 }
                    if (segmentTerminals == 2) then
                        terminals2[#terminals2 + 1] = { #result.models, model.swapTerminals and 0 or 1 }
                    end
                end
                			    
                result.models[#result.models + 1] = {
	                id = epochFilename(config, path .. model.platform),
	                transf = transf.rotZYXTransl(transf.degToRad(rotateZ, 0, 0), vec3.new(x, y, z))
                }
                
                if (config.platformRoofing and model.roof) then
	                result.models[#result.models + 1] = {
		                id = epochFilename(config, path .. model.roof),
		                transf = transf.rotZYXTransl(transf.degToRad(rotateZ, 0, 0), vec3.new(x, y, z))
	                }
                end
                
                if (model.assets) then
	                result.models[#result.models + 1] = {
		                id = epochFilename(config, path .. model.assets),
		                transf = transf.rotZYXTransl(transf.degToRad(rotateZ, 0, 0), vec3.new(x, y, z))
	                }
                end
            end
                        
            if not ((i == 1 or i == #segments) and hasEndparts) then
	            yOffset = yOffset + segmentLength
	        end
        end
    end
    return { terminals1, terminals2 }
end

local function buildDeadEndTrack(config, trackDefinition)
    local x = trackDefinition.x
    local direction = trackDefinition.direction
    local height = trackDefinition.height
    local length = config.stationLength / 2 + TRACK_EXTENSION + (config.lengthEndSegment or 0)
    local length2 = 10
    
    config.stationY = -length < config.stationY and -length or config.stationY
    config.stationYEnd = length > config.stationYEnd and length or config.stationYEnd

    local yOffset = direction * length            
    local yDistance = direction * length2
    
    local yOffset2 = direction * (length - length2)           
    local yDistance2 = direction * (2 * length - length2)

    local trackDeadEnd = trackDefinition.track.trackDeadEnd
    local deadEndTrack = trackDefinition.track.deadEndTrack

    -- short additional track for raised stations to prevent terrain allignment
    -- under the station when the user connects his tracks...
    local yExtend = direction * PLANUM_AREA

    local edges = { }               -- the track segments
    local snapNodes = { }           -- track endpoints to the outside world
    local vehicleNodes = {}         -- stopping point of the train

    if (deadEndTrack and direction > 0) or (trackDeadEnd and direction < 0) then
        edges[#edges + 1] = { { x,  -yOffset, height }, { 0, yDistance, 0 } }
        vehicleNodes[#vehicleNodes + 1] = #edges    
        edges[#edges + 1] = { { x, -yOffset2, height }, { 0, yDistance, 0 } }

        edges[#edges + 1] = { { x, -yOffset2, height }, { 0, yDistance2, 0 } }
        edges[#edges + 1] = { { x,   yOffset, height }, { 0, yDistance2, 0 } }
    elseif height ~= 0 then
        -- prevent terrain allignment
        edges[#edges + 1] = { { x, -yOffset - yExtend, height }, { 0, yExtend, 0 } }
        edges[#edges + 1] = { { x, -yOffset,           height }, { 0, yExtend, 0 } }
    end
    if (deadEndTrack and direction < 0) or (trackDeadEnd and direction > 0) then
        edges[#edges + 1] = { { x,  -yOffset, height }, { 0, yDistance2, 0 } }
        edges[#edges + 1] = { { x,  yOffset2, height }, { 0, yDistance2, 0 } }

        vehicleNodes[#vehicleNodes + 1] = #edges    
        edges[#edges + 1] = { { x,  yOffset2, height }, { 0, yDistance, 0 } }
        edges[#edges + 1] = { { x,   yOffset, height }, { 0, yDistance, 0 } }
    elseif height ~= 0 then
        -- prevent terrain allignment
        edges[#edges + 1] = { { x, yOffset,           height }, { 0, yExtend, 0 } }
        edges[#edges + 1] = { { x, yOffset + yExtend, height }, { 0, yExtend, 0 } }
    end

    if not ((deadEndTrack and direction > 0) or (trackDeadEnd and direction < 0)) then
        snapNodes[#snapNodes+1] = 0 
    end
    if not ((deadEndTrack and direction < 0) or (trackDeadEnd and direction > 0)) then
        snapNodes[#snapNodes+1] = #edges - 1
    end
  
    trackDefinition.edges = edges
    trackDefinition.snapNodes = snapNodes
    trackDefinition.vehicleNodes = vehicleNodes
    trackDefinition.slowTrackType = true
end

local function buildTrack(config, trackDefinition)
    if trackDefinition.track.trackDeadEnd or trackDefinition.track.deadEndTrack then
        buildDeadEndTrack(config, trackDefinition)
        return
    end
    
    local x = trackDefinition.x
    local direction = trackDefinition.direction
    local height = config.height
    local length = config.availableSizes[trackDefinition.sizeIndex]
    length = length * config.segmentLength
    length = length / 2 + TRACK_EXTENSION + (config.lengthEndSegment or 0)

    config.stationY = -length < config.stationY and -length or config.stationY
    config.stationYEnd = length > config.stationYEnd and length or config.stationYEnd

    local yOffset = direction * length            
    local yDistance = yOffset
    local yPos = yDistance
    
    -- short additional track for raised stations to prevent terrain allignment
    -- under the station when the user connects his tracks...
    local yExtend = direction * PLANUM_AREA
    
    local edges = { }               -- the track segments
    local snapNodes = { }           -- track endpoints to the outside world
    local vehicleNodes = {}         -- stopping point of the train

    local splitTrack = trackDefinition.split
      
    local turnout = trackDefinition.turnout
    local turnoutLength
    local turnoutLength2

    if turnout then
        turnoutLength = direction * turnout.length
        turnoutLength2 = turnoutLength / 2
        yDistance = (yOffset - turnoutLength2) / 2
        yPos = yOffset - yDistance
        splitTrack = true
    elseif splitTrack then
        yDistance = yOffset / 2
        yPos = yDistance
        trackDefinition.slowTrackType = true
    end
    
    trackDefinition.yStart = -yOffset
    trackDefinition.yEnd = yOffset
    
    if height ~= 0 then
        -- prevent terrain allignment
        trackDefinition.yStart = -yOffset - yExtend
        edges[#edges + 1] = { { x, -yOffset - yExtend, height }, { 0, yExtend, 0 } }
        edges[#edges + 1] = { { x, -yOffset,           height }, { 0, yExtend, 0 } }
    end

    edges[#edges + 1] = { { x, -yOffset, height }, { 0, yDistance, 0 } }

    if splitTrack then      
        vehicleNodes[#vehicleNodes + 1] = #edges
        edges[#edges + 1] = { { x, -yPos, height }, { 0, yDistance, 0 } }
        edges[#edges + 1] = { { x, -yPos, height }, { 0, yDistance, 0 } }
    end
         
    if turnout then
        edges[#edges + 1] = { { x, -turnoutLength2, height }, { 0, yDistance, 0 } }
        edges[#edges + 1] = { { x, -turnoutLength2, height }, { 0, turnoutLength, 0 } }
        edges[#edges + 1] = { { x,  turnoutLength2, height }, { 0, turnoutLength, 0 } }
        edges[#edges + 1] = { { x,  turnoutLength2, height }, { 0, yDistance, 0 } }
    else
        if not splitTrack then
	        vehicleNodes[#vehicleNodes + 1] = #edges
        end         
        edges[#edges + 1] = { { x, 0, height }, { 0, yDistance, 0 } }
        edges[#edges + 1] = { { x, 0, height }, { 0, yDistance, 0 } }
    end
        
    if splitTrack then
        vehicleNodes[#vehicleNodes + 1] = #edges
        edges[#edges + 1] = { { x,  yPos, height }, { 0, yDistance, 0 } }
        edges[#edges + 1] = { { x,  yPos, height }, { 0, yDistance, 0 } }
    end

    edges[#edges + 1] = { { x, yOffset, height }, { 0, yDistance, 0 } }

    if height ~= 0 then
        -- prevent terrain allignment
        trackDefinition.yEnd = yOffset + yExtend
        edges[#edges + 1] = { { x, yOffset,           height }, { 0, yExtend, 0 } }
        edges[#edges + 1] = { { x, yOffset + yExtend, height }, { 0, yExtend, 0 } }
    end

    snapNodes[#snapNodes+1] = 0 
    snapNodes[#snapNodes+1] = #edges - 1

	if turnout and not trackDefinition.transit then
	    local turnoutEdges
        if turnout.crossover then
            turnoutEdges = edgeutil.makeCrossoverEdges(
                x, -turnoutLength2, turnoutLength2, 
                height, config.trackDistance, turnout.leftHanded
            )
        elseif turnout.left then
            -- TODO
        elseif turnout.right then
            -- TODO
        end
        if turnoutEdges then
            for teIdx = 1, #turnoutEdges do
                edges[#edges + 1] = turnoutEdges[teIdx]
            end  
            trackDefinition.slowTrackType = true      
        end
	end

    trackDefinition.edges = edges
    trackDefinition.snapNodes = snapNodes
    trackDefinition.vehicleNodes = vehicleNodes
end

local function buildTerminals(config, result, trackDefinition)
    local terminals = trackDefinition.terminals
    if not terminals then
        return
    end
    
    local reverse = trackDefinition.direction < 0
    local vehicleNodes = trackDefinition.vehicleNodes
    local terminalCount = #terminals
    if terminalCount > 0 then
        local vehicleNode = vehicleNodes[1]
        if trackDefinition.split or trackDefinition.turnout then
            local from = reverse and terminalCount / 2 + 1 or 1
            local to = reverse and terminalCount or terminalCount / 2
            result.terminalGroups[#result.terminalGroups + 1] = { 
                terminals = table.pack(table.unpack(terminals, from, to)), 
                vehicleNodeOverride = vehicleNode 
            }
            from = reverse and 1 or terminalCount / 2 + 1
            to = reverse and terminalCount / 2 or terminalCount
            vehicleNode = vehicleNodes[2]
            terminals = table.pack(table.unpack(terminals, from, to))
        end
        result.terminalGroups[#result.terminalGroups + 1] = { 
            terminals = terminals, vehicleNodeOverride = vehicleNode 
        }
    end
end

local function alignTrackTerrain(config, result, trackDefinition)
    local groundFaces = result.groundFaces
    local height = trackDefinition.height
    local faces = { }
    local x = trackDefinition.x
    
    local x1 = x - config.trackDistance / 2
    local x2 = x + config.trackDistance / 2

    if trackDefinition.platformWidth then
        if trackDefinition.platformWidth < 0 then
            x1 = x1 + trackDefinition.platformWidth
        else
            x2 = x2 + trackDefinition.platformWidth
        end
    end        
       
    if height ~= 0 then
        local offset = 2 * PLANUM_AREA
        faces[#faces + 1] = {
            { x1, trackDefinition.yStart - offset, height },
            { x2, trackDefinition.yStart - offset, height },
            { x2, trackDefinition.yStart + offset, height },
            { x1, trackDefinition.yStart + offset, height },
        }    
        faces[#faces + 1] = {
            { x1, trackDefinition.yEnd - offset, height },
            { x2, trackDefinition.yEnd - offset, height },
            { x2, trackDefinition.yEnd + offset, height },
            { x1, trackDefinition.yEnd + offset, height },
        }    
    end
    if height <= 0 or config.alignTerrain then
        if x1 < 0 then
            faces[#faces + 1] = {
                { x1, trackDefinition.yStart, height },
                { x1, trackDefinition.yEnd, height },
                { x2, trackDefinition.yEnd, height },
                { x2, trackDefinition.yStart, height },
            }
        else
            faces[#faces + 1] = {
                { x1, trackDefinition.yStart, height },
                { x2, trackDefinition.yStart, height },
                { x2, trackDefinition.yEnd, height },
                { x1, trackDefinition.yEnd, height },
            }
        end
    end

    -- for i = 1, #faces do
        -- groundFaces[#groundFaces + 1] = { face = faces[i], modes = { { type = "FILL", key = "industry_gravel_small_01" } } }
        -- groundFaces[#groundFaces + 1] = { face = faces[i], modes = { { type = "STROKE_OUTER", key = "building_paving" } } }        
    -- end
    --result.terrainAlignmentLists[#result.terrainAlignmentLists + 1] = { type = "EQUAL", faces = faces, } 
end

local function buildTracks(config, result, trackDefinitions)
    -- first pass: build tracks and turnouts
    local platformNumber = 0
    for i = 1, #trackDefinitions do
        local trackDef = trackDefinitions[i]
        if trackDef.platformLayout and config.overrideSizeIndexFn then
            platformNumber = platformNumber + 1
            trackDef.sizeIndex = config.overrideSizeIndexFn(config, platformNumber, trackDef.sizeIndex)
            trackDef.sizeIndex = trackDef.sizeIndex < 1 and 1 or trackDef.sizeIndex
            if trackDef.tracks == 2 then
                trackDefinitions[i + 1].sizeIndex = trackDef.sizeIndex
            end
        end
        local turnout = trackDef.turnout
        if turnout then
            if turnout.leftHanded then
                local prevTrackDef = i > 1 and trackDefinitions[i - 1]
                if prevTrackDef and prevTrackDef.transit then
                    if not prevTrackDef.turnout then
                        prevTrackDef.turnout = turnout
                        buildTrack(config, prevTrackDef)   -- rebuild last track
                    else
                        trackDef.turnout = nil
                    end 
                else
                    trackDef.turnout = nil
                end
            else
                -- rightHanded
                if not trackDef.transit then
                    local nextTrackDef = i < #trackDefinitions and trackDefinitions[i + 1]
                    if nextTrackDef and nextTrackDef.transit then
                        nextTrackDef.turnout = turnout
                    else
                        trackDef.turnout = nil                
                    end
                end
            end
        end
        buildTrack(config, trackDef)
        alignTrackTerrain(config, result, trackDef)        
    end
    
    -- second pass: persist tracks, build platforms
    local totalEdgeCount = getTotalEdgeCount(result)
    local terminals
    
    for i = 1, #trackDefinitions do
        local trackDef = trackDefinitions[i]
        
        if config.debug then 
            local txt = trackDef.turnout and " turnout" or (trackDef.split and " split" or "")
            if trackDef.turnout then
                txt = txt .. (trackDef.turnout.leftHanded and " left-handed" or " right-handed")    
                txt = txt .. (trackDef.transit and " transit" or "")  
            end
            print("        building track #" .. i .. " @ x = " .. trackDef.x .. txt)      
        end                
                    
        result.edgeLists[#result.edgeLists + 1] = { 
	        type = "TRACK",
	        alignTerrain = trackDef.height == 0 or config.alignTerrain,
	        params = {
		        type = trackDef.slowTrackType and config.deadEndTrackType or config.trackType,
		        catenary = config.catenary,
	        },
	        edges = trackDef.edges,
	        snapNodes = trackDef.snapNodes,
        }
        
        for j = 1, #trackDef.vehicleNodes do
            trackDef.vehicleNodes[j] = trackDef.vehicleNodes[j] + totalEdgeCount
        end        
        totalEdgeCount = totalEdgeCount + #trackDef.edges

        if trackDef.platformLayout then
            terminals = buildPlatform(config, result, trackDef, trackDef.platformLayout)
            trackDef.terminals = terminals[1]
            if trackDef.track.tracks > 1 and #trackDefinitions > i then
                trackDefinitions[i + 1].terminals = terminals[2]
            end
        end
        buildTerminals(config, result, trackDef)
        if trackDef.lastOddPlatformLayout then
            buildPlatform(config, result, trackDef, trackDef.lastOddPlatformLayout)
        end
    end
end

-- build the whole layout
local function buildLayout(config, result, trackLayout)
    local trackDistance = config.trackDistance
    local numberOfSegments = config.numberOfSegments

    local stationLength = numberOfSegments * config.segmentLength
    config.stationLength = stationLength
    
    if config.debug then 
        print("    buildLayout() with " .. config.numberOfTracks .. " tracks...")      
    end

    local trackDefinitions = { }
    local platformCount = 0
    local trackCount = 0
    local transitTrackCount = 0
    
    local x = 0
    
    for i = 1, #trackLayout do
        local track = trackLayout[i]
        local numberOfTracks = track.numberOfTracks
        numberOfTracks = math.floor((type(numberOfTracks) == "function" and numberOfTracks(config) or numberOfTracks) + 0.5)

        if config.debug then 
            print("      Layout #" .. i .. ": desired numberOfTracks = " .. numberOfTracks)      
        end

        -- Reduce the number of tracks of a two-sided platform until the maximum number of tracks to be built fits
        while track.tracks > 1 and (trackCount + numberOfTracks >= config.numberOfTracks) do
            numberOfTracks = numberOfTracks - 1
        end      

        if numberOfTracks > 0 and (trackCount + numberOfTracks <= config.numberOfTracks or track.transit) then

            if config.debug then 
                print("      Layout #" .. i .. ": realized numberOfTracks = " .. numberOfTracks .. " with " .. track.tracks .. " tracks/platform")      
            end

            local platformLayout = track.platformLayout
            local width = platformLayout.width

            if i == 1 then 
                if width < 0 then
                    config.stationX = width + (platformLayout.moveX or 0)   
                else                 
                    config.stationX = -trackDistance / 2 
                end
                x = width < 0 and width or -trackDistance
            end
            x = x + (track.moveX or 0)

            -- left/right door opening depends on the direction of the track:
            local direction = width < 0 and -1 or 1

            local trackDef = { }

            -- firstTrack-Options
            trackDef.split = track.splitFirstTrack
            trackDef.turnout = track.turnoutFirstTrack
            
            for j = 1, numberOfTracks, track.tracks do
                for k = 1, track.tracks do
                    trackDef.direction = direction
                    trackDef.track = track                    
                    trackDef.transit = track.transit
                    trackDef.height = track.height or config.baseHeight
                    trackDef.sizeIndex = config.sizeIndex
                    
                    if k == 1 then
                        x = x + (width < 0 and -width or trackDistance)

                        trackDef.platformLayout = platformLayout
                        trackDef.platformWidth = width
                        trackDef.tracks = track.tracks
                        platformCount = platformCount + 1
                    else
                        x = x + (width > 0 and width or trackDistance)    
                    end

                    trackDef.x = x
                    trackDefinitions[#trackDefinitions + 1] = trackDef
                    trackDef = { }
                    
                    if track.transit then
                        transitTrackCount = transitTrackCount + 1
                    else
                        trackCount = trackCount + 1
                    end
                    direction = -direction
                end                
            end
            trackDef = trackDefinitions[#trackDefinitions]
            -- lastTrack-Options
            trackDef.split = trackDef.split or track.splitLastTrack
            trackDef.turnout = trackDef.turnout or track.turnoutLastTrack
            config.lastTrack = track    
        else
            if config.debug then 
                print("      Layout #" .. i .. ": skipping numberOfTracks = " .. numberOfTracks)      
            end    
        end        
    end
    
    local trackDef = trackDefinitions[#trackDefinitions]
    config.stationXEnd = trackDef.x 
    local platformLayout = trackLayout.lastOddPlatformLayout
    if (trackDef.transit or config.numberOfTracks % 2 ~= 0) and platformLayout then
        trackDef.lastOddPlatformLayout = platformLayout
        config.stationXEnd = config.stationXEnd + platformLayout.width + (platformLayout.moveX or 0) 
    elseif trackDef.platformWidth and trackDef.platformWidth > 0 then
        config.stationXEnd = config.stationXEnd + trackDef.platformWidth + (trackDef.platformLayout.moveX or 0) 
    else
        config.stationXEnd = config.stationXEnd + trackDistance / 2
    end
    config.trackCount = trackCount
    config.transitTrackCount = transitTrackCount
    config.numberOfPlatforms = platformCount
     
    buildTracks(config, result, trackDefinitions)
end

-- align the terrain
local function alignTerrain(config, result)
    local baseHeight = config.baseHeight
    if baseHeight > 0 then
        return
    end
    
    local stationX = config.stationX - 1
    local stationXEnd = config.stationXEnd + 1       
    local halfStationLength = config.stationLength / 2 + TRACK_EXTENSION + (config.lengthEndSegment or 0)

    local face = { 
            { stationX,    -halfStationLength, baseHeight},  
            { stationXEnd, -halfStationLength, baseHeight}, 
            { stationXEnd,  halfStationLength, baseHeight}, 
            { stationX,     halfStationLength, baseHeight}   
    }	    
    --result.terrainAlignmentLists[#result.terrainAlignmentLists + 1] = { type = "EQUAL", faces = { face } } 
end

local function resetResult(result)
    result.models = { }                 -- the 3D platform models
    result.terminalGroups = { }         -- connections platforms and trains
    result.edgeLists = { }              -- tracks and snapnodes
    result.terrainAlignmentLists = { }
    result.groundFaces = { }
end

local function resetCalculatedConfigValues(config)
    config.trackCount = 0
    config.transitTrackCount = 0
    config.numberOfPlatforms = 0
    config.stationX = 0
    config.stationXEnd = 0
    config.stationY = 0
    config.stationYEnd = 0
    config.currentX = 0
end

----------------------------------------------------------------------------
-- public function to do all the work...

function StationBuilder.buildStation(params, config, result) 
    if config.debug then 
        print("-----------------------------")
        print("StationBuilder.buildStation()")      
    end
	
	config.height = config.trackTypeHeights[params.tom_trackType + 1]

    config.layoutIndex = config.layoutIndex or 1
    config.layoutIndex = (config.layoutIndex > #config.trackLayouts) and #config.trackLayouts or config.layoutIndex
    
    -- initialize result tables and calculated values...
    resetResult(result)
    resetCalculatedConfigValues(config)

    -- build tracks and platforms. Calculate dimensions and numbers...
    buildLayout(config, result, config.trackLayouts[config.layoutIndex]) 
    
    if config.layoutStartingFn then
        -- Huh - someone wants to build some parts of the station first!
        -- Discard all result tables...
        resetResult(result)
        -- Let him do his work. He has access to the final dimensions and numbers...
        config.layoutStartingFn(params, config, result)
        -- Discard all calculated dimensions and numbers...
        resetCalculatedConfigValues(config)
        -- ...and build tracks and platforms again...
        buildLayout(config, result, config.trackLayouts[config.layoutIndex]) 
    end

    alignTerrain(config, result)

    if config.layoutFinishedFn then
        -- After all tracks and platforms are in their place, 
        -- let's add buildings, street connections and other fancy stuff...
        config.layoutFinishedFn(params, config, result)
    end

    if config.debug then 
        print("StationBuilder.buildStation()")      
        print("-----------------------------")
    end

	return result
end

return StationBuilder

