local transf = require "transf"

local vec2 = require "snowball/common/vec2_1"
local vec3 = require "snowball/common/vec3_1"
local mat2 = require "snowball/common/mat2_1"
local mat3 = require "snowball/common/mat3_1"

local function isTable(tab) return type(tab) == 'table' end

local fences = {}

fences.faceLeft = 0
fences.faceRight = 1

fences.modeLinear = 0
fences.modeCurve = 1
fences.modeArc = 2
fences.modeFinish = 3

fences.collisionOn = 0
fences.collisionAuto = 1
fences.collisionOff = 2

fences.fences = {}

function fences.createParams()
    
    local params = {}
    
	params[#params + 1] = {
		key = "snowball_fences_mode",
		name = _("snowball_fences_mode"),
		values = {_("snowball_fences_linear"), _("snowball_fences_curve"), _("snowball_fences_arc"), _("snowball_fences_build")},
		defaultIndex = 0
	}
	
    params[#params + 1] = {
        key = "snowball_fences_face",
        name = _("snowball_fences_face"),
        values = {           
            _("snowball_fences_face_left"),
            _("snowball_fences_face_right"),
        },       
        defaultIndex = 0
    }
    
    params[#params + 1] = {
        key = "snowball_fences_collision",
        name = _("snowball_fences_collision"),
        values = {
            _("snowball_fences_collision_on"),
            _("snowball_fences_collision_auto"),
            _("snowball_fences_collision_off"),
        
        },
        defaultIndex = 0
    }
    
    local offsets = {}
    for i = 0, 100 do
        offsets[#offsets + 1] = tostring(i / 10)
    end
    
    params[#params + 1] = {
        key = "snowball_fences_offset",
        name = _("snowball_fences_offset"),
        values = offsets,
        uiType = "SLIDER",
        defaultIndex = 0
    }

    params[#params + 1] = {
        key = "snowball_fences_adjust",
        name = _("snowball_fences_adjust"),
        values = { "0", "1" },  
        uiType = "CHECKBOX",  
        defaultIndex = 1
    }

    params[#params + 1] = {
        key = "snowball_fences_stepped",
        name = _("snowball_fences_stepped"),
        values = { "diagonal", "stepped" },  
        uiType = "CHECKBOX",
        defaultIndex = 1
    }
    
    return params
end

function fences.createParamsAdvanced(types, offsetCoeffs, fixMode)
    
    local params = {}
    
	if fixMode == nil then
		params[#params + 1] = {
			key = "snowball_fences_mode",
			name = _("snowball_fences_mode"),
			values = {_("snowball_fences_linear"), _("snowball_fences_curve"), _("snowball_fences_arc"), _("snowball_fences_build")},
			defaultIndex = 0
		}
    end
	    
	params[#params + 1] = {
        key = "snowball_fences_type",
        name = _("snowball_fences_type"),
		tooltip = _("snowball_fences_type_tip"),
		uiType = "ICON_BUTTON",
        values = types,
        defaultIndex = 0
    }
	
	
    params[#params + 1] = {
        key = "snowball_fences_face",
        name = _("snowball_fences_face"),
        values = {           
            _("snowball_fences_face_left"),
            _("snowball_fences_face_right"),
        },       
        defaultIndex = 0
    }
    
    params[#params + 1] = {
        key = "snowball_fences_collision",
        name = _("snowball_fences_collision"),
        values = {
            _("snowball_fences_collision_on"),
            _("snowball_fences_collision_auto"),
            _("snowball_fences_collision_off"),
        
        },
        defaultIndex = 0
    }

    params[#params + 1] = {
        key = "snowball_fences_adjust",
        name = _("snowball_fences_adjust"),
        values = { "0", "1" },  
        uiType = "CHECKBOX",
        defaultIndex = 1
    }

    params[#params + 1] = {
        key = "snowball_fences_stepped",
        name = _("snowball_fences_stepped"),
        values = { "diagonal", "stepped" },  
        uiType = "CHECKBOX",
        defaultIndex = 0
    }
    
	if offsetCoeffs[1] ~= 0 then
		local offsets = {}
		for i = -10, 10 do
			offsets[#offsets + 1] = tostring(i * offsetCoeffs[1])
		end
		
		params[#params + 1] = {
        key = "snowball_fences_offset_x",
		name = _("snowball_fences_offset_x"),
		tooltip = _("snowball_fences_offset_x_tip"),
		uiType = "SLIDER",
		values = offsets,
		defaultIndex = 10
		}
	end
	if offsetCoeffs[2] ~= 0 then
		local offsets = {}
		for i = -10, 10 do
			offsets[#offsets + 1] = tostring(i * offsetCoeffs[2])
		end
		
		params[#params + 1] = {
        key = "snowball_fences_offset_y",
		name = _("snowball_fences_offset_y"),
		tooltip = _("snowball_fences_offset_y_tip"),
		uiType = "SLIDER",
		values = offsets,
		defaultIndex = 10
		}
	end
	if offsetCoeffs[3] ~= 0 then
		local offsets = {}
		for i = -10, 10 do
			offsets[#offsets + 1] = tostring(i * offsetCoeffs[3])
		end
		
		params[#params + 1] = {
        key = "snowball_fences_offset_z",
		name = _("snowball_fences_offset_z"),
		tooltip = _("snowball_fences_offset_z_tip"),
		uiType = "SLIDER",
		values = offsets,
		defaultIndex = 10
		}
	end
    
    return params
end

function fences.getOffsetFromParam(offset)
    return (offset or 0) / 10
end

function fences.getOffsetFromParamAdvanced(params,offsetCoeffs)
	
	local offset = {0,0,0}
	if params.snowball_fences_offset_x then
		offset[1] = params.snowball_fences_offset_x
	end
	if params.snowball_fences_offset_y then
		offset[2] = params.snowball_fences_offset_y
	end
	if params.snowball_fences_offset_z then
		offset[3] = params.snowball_fences_offset_z
	end
	
	offset[1] = - 10 * offsetCoeffs[1] + offset[1] * offsetCoeffs[1]
	offset[2] = - 10 * offsetCoeffs[2] + offset[2] * offsetCoeffs[2]
	offset[3] = - 10 * offsetCoeffs[3] + offset[3] * offsetCoeffs[3]


    return offset
end

function fences.middle(a, b, fence, invertedModel, stepped, result)
    
    local fenceLength = fence.length
    
	local a1 = { a[1],a[2],a[3] }
	local b1 = { b[1],b[2],b[3] }
	
	if stepped == 1 then
		if a1[3] > b1[3] then
			a1[3] = b1[3]
		else
			b1[3] = a1[3]
		end
	end
	
    local segment = vec3.sub(b1, a1)
    local segmentLength = vec3.length(segment)
    local transformedX = vec3.mul(segmentLength / fenceLength, vec3.normalize(segment))
    local transformedY = nil
	local trafomatrix = nil
	
	
	
	if not invertedModel then
		transformedY = vec3.normalize({segment[2], -segment[1], 0.0})
		trafomatrix = mat3.toTransf(mat3.affine(transformedX, transformedY), a1)
	else
		transformedY = vec3.normalize({-segment[2], segment[1], 0.0})
		trafomatrix = mat3.toTransf(mat3.affine(vec3.mul(-1,transformedX), vec3.mul(-1,transformedY)), b1)
	end
	
    if fence.middle then
        if isTable(fence.middle) then
			result.models[#result.models + 1] = {
				id = fence.middle[math.random(1, #fence.middle)],
				transf = trafomatrix,
			}
		else 
			result.models[#result.models + 1] = {
				id = fence.middle,
				transf = trafomatrix
			}
		end
    end
    
    if fence.decoration then
        result.models[#result.models + 1] = {
            id = fence.decoration[math.random(1, #fence.decoration)],
            transf = transf.rotZTransl(math.random() * math.pi * 2.0, {
                x = b[1] + (math.random() - 0.5) * 0.2 * fence.length,
                y = b[2] + (math.random() - 0.5) * 0.2 * fence.length,
                z = b[3]
            })
        }
    end
end

function fences.post(previous, position, next, fence, result)
    
    if not fence.post or not previous or not next then
        return
    end
    
    local rotation = math.atan2(next[2] - previous[2], next[1] - previous[1])
    
	
	if isTable(fence.post) then
		result.models[#result.models + 1] = {
			id = fence.post[math.random(1, #fence.post)],
			transf = transf.rotZTransl(rotation, {x = position[1], y = position[2], z = position[3]})
		}
	else 
		result.models[#result.models + 1] = {
			id = fence.post,
			transf = transf.rotZTransl(rotation, {x = position[1], y = position[2], z = position[3]})
		}
	end

end

function fences.createLine(skipFirstPost, from, to, fence, face, adjust, stepped, offsetZ, invertedModel, result)
    
    if not result.models then
        result.models = {}
    end
    
    local modelLength = fence.length
    local fenceVector = vec3.sub(to, from)
    local fenceLength = vec3.length(fenceVector)
    local segmentCount = math.floor(fenceLength / modelLength + 0.5)
    
    if segmentCount == 0 then
        segmentCount = 1
    end
    
    local segment = vec3.mul(1 / segmentCount, fenceVector)
    
    local p = {
        nil,
        from
    }

	if adjust == 1 then
		p[2][3] = game.interface.getHeight({p[2][1], p[2][2]}) + offsetZ
	end

    for j = 1, segmentCount do
        
        p[3] = vec3.add(from, vec3.mul(j, segment))        
        
        if adjust== 1 then
            p[3][3] = game.interface.getHeight({p[3][1], p[3][2]}) + offsetZ
        else
            p[3][3] = from[3] + j / segmentCount * (to[3] - from[3])
        end

        if j == 1 and skipFirstPost then
            p[1] = nil
        elseif j == 1 and skipFirstPost == false then
			p[1] = vec3.sub(p[2],segment)
		end
        
        if face == fences.faceLeft then
            fences.post(p[3], p[2], p[1], fence, result)
            fences.middle(p[3], p[2], fence, invertedModel, stepped, result)
            
            if j == segmentCount then
                fences.post(p[3], p[3], p[2], fence, result)
            end
        
        else
            fences.post(p[1], p[2], p[3], fence, result)
            fences.middle(p[2], p[3], fence, invertedModel, stepped, result)
            
            if j == segmentCount then
                fences.post(p[2], p[3], p[3], fence, result)
            end
        
        end

        p[1] = p[2]
        p[2] = p[3]

    end
    
    return from
end

function fences.bezier(t, b0, b1, b2, b3)
    return
        t * t * t * (-1 * b0 + 3 * b1 - 3 * b2 + b3) +
        t * t * (3 * b0 - 6 * b1 + 3 * b2) +
        t * (-3 * b0 + 3 * b1) +
        b0
end

function fences.createCurve(skipFirstPost, previous, from, to, fence, face, adjust, stepped, offsetZ, invertedModel, result)
    
    if not result.models then
        result.models = {}
    end
    
    local modelLength = fence.length
    local fenceLength = vec3.distance(from, to)
    local segmentCount = math.floor(fenceLength / modelLength + 0.5)
    
    if segmentCount == 0 then
        segmentCount = 1
    end
    
    local b0 = from
    local b3 = to;
    
    local v1 = vec2.sub(from, previous)
    local v2 = vec2.sub(b3, b0)
    
    local nv1 = vec2.normalize(v1)
    local nv2 = vec2.normalize(v2)
    
    local b1 = vec2.add(b0, vec2.mul(0.3333 * vec2.length(v2), nv1));
    local c1 = vec2.sub(b1, b0)
    
    local a = math.pi + 2 * vec2.angle(v1, v2);
    local b2 = vec2.add(b3, {
        c1[1] * math.cos(a) - c1[2] * math.sin(a),
        c1[1] * math.sin(a) + c1[2] * math.cos(a)
    })
    
    local p = {
        nil,
        from
    }

	if adjust == 1 then
		p[2][3] = game.interface.getHeight({p[2][1], p[2][2]}) + offsetZ
	end

    
    for j = 1, segmentCount do
        
        p[3] = {
            fences.bezier(1.0 / segmentCount * j, b0[1], b1[1], b2[1], b3[1]),
            fences.bezier(1.0 / segmentCount * j, b0[2], b1[2], b2[2], b3[2])
        }        
        
        if adjust == 1 then
            p[3][3] = game.interface.getHeight({p[3][1], p[3][2]}) + offsetZ
        else
            p[3][3] = from[3] + j / segmentCount * (to[3] - from[3])
        end

        if j == 1 and skipFirstPost then
            p[1] = nil
        end
        
        if face == fences.faceLeft then
            fences.post(p[3], p[2], p[1], fence, result)
            fences.middle(p[3], p[2], fence, invertedModel, stepped, result)
            
            if j == segmentCount then
                fences.post(p[3], p[3], p[2], fence, result)
            end
        
        else
            fences.post(p[1], p[2], p[3], fence, result)
            fences.middle(p[2], p[3], fence, invertedModel, stepped, result)
            
            if j == segmentCount then
                fences.post(p[2], p[3], p[3], fence, result)
            end
        
        end
        
        p[1] = p[2]
        p[2] = p[3]
    
    end
    
    return b2
end

function fences.createArc(skipFirstPost, previous, from, to, fence, face, adjust, stepped, offsetZ, invertedModel, result)
    
    if not result.models then
        result.models = {}
    end
    
    local p1 = previous
    local p2 = from
    local p3 = to
    local t = vec2.sub(p2, p1);
    local o = {t[2], -t[1]};
    
    --Nenner aus der umgestellten Kreisgleichung
    local n = (2 * (p3[1] - p2[1]) * o[1] + 2 * (p3[2] - p2[2]) * o[2])

    if math.abs(n) < 0.1 then
        return fences.createLine(skipFirstPost, from, to, fence, face, adjust, stepped, offsetZ, invertedModel, result)
    end

    local s = ((p3[1] - p2[1]) * (p3[1] - p2[1]) + (p3[2] - p2[2]) * (p3[2] - p2[2])) / n
    
    if math.abs(s) < 0.1 then
        return fences.createLine(skipFirstPost, from, to, fence, face, adjust, stepped, offsetZ, invertedModel, result)
    end 

    local c = vec2.add(p2, vec2.mul(s, o))
    local r = vec2.distance(c, p2);
    local cp2 = vec2.sub(p2, c)
    local cp3 = vec2.sub(p3, c);
    local angle = vec2.angle(cp2, cp3)
    local checkAngle = vec2.angle(cp2, t)
    
    if checkAngle < 0 and angle > 0 then
        angle = angle - math.pi * 2;
    elseif checkAngle > 0 and angle < 0 then
        angle = angle + math.pi * 2;
    end
    
    local segmentCount = math.max(1, math.floor(math.abs(angle * r) / fence.length + 0.5))
    local delta = angle / segmentCount;
    local p = {
        nil,
        from
    }

	if adjust == 1 then
		p[2][3] = game.interface.getHeight({p[2][1], p[2][2]}) + offsetZ
	end

    
    for j = 1, segmentCount do
        
        local a = j * delta
        p[3] = {
            cp2[1] * math.cos(a) - cp2[2] * math.sin(a) + c[1],
            cp2[1] * math.sin(a) + cp2[2] * math.cos(a) + c[2]
        }        
       
        if adjust == 1 then
            p[3][3] = game.interface.getHeight({p[3][1], p[3][2]}) + offsetZ
        else
            p[3][3] = from[3] + j / segmentCount * (to[3] - from[3])
        end

        if j == 1 and skipFirstPost then
            p[1] = nil
        end
        
        if face == fences.faceLeft then
            fences.post(p[3], p[2], p[1], fence, result)
            fences.middle(p[3], p[2], fence, invertedModel, stepped, result)
            
            if j == segmentCount then
                fences.post(p[3], p[3], p[2], fence, result)
            end
        
        else
            fences.post(p[1], p[2], p[3], fence, result)
            fences.middle(p[2], p[3], fence, invertedModel, stepped, result)
            
            if j == segmentCount then
                fences.post(p[2], p[3], p[3], fence, result)
            end
        
        end
        
        p[1] = p[2]
        p[2] = p[3]
    
    end
    
    return p[1]
end

function fences.preview(params, model)
    
    local offset = fences.getOffsetFromParam(params.snowball_fences_offset)
    
    return {
        models = {
            {
                id = model,
                transf = {1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, offset, 0, 1}
            }
        },
        cost = 10,
        bulldozeCost = 10,
        maintenanceCost = 0,
        terrainAlignmentLists = {
            {
                type = "EQUAL",
                faces = {}
            }
        }
    }

end

function fences.previewAdvanced(params, offsetCoeffs, model)
    
    local offsets = fences.getOffsetFromParamAdvanced(params, offsetCoeffs)
    
    return {
        models = {
            {
                id = model,
                transf = {1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, offsets[1], offsets[2], offsets[3], 1}
            }
        },
        cost = 10,
        bulldozeCost = 10,
        maintenanceCost = 0,
        terrainAlignmentLists = {
            {
                type = "EQUAL",
                faces = {}
            }
        }
    }

end

function fences.build(skipFirstPost, previous, from, to, fence, face, adjust, stepped, collision, segmentType, offsetZ, invertedModel)
    
    if not fence or not from or not to or vec2.distanceSquared(from, to) < 0.01 then
        return nil, from
    end
    
    local constructionId = "asset/snowball_fences_fence_on.con"
    if collision == fences.collisionAuto then
        constructionId = "asset/snowball_fences_fence_auto.con"
    elseif collision == fences.collisionOff then
        constructionId = "asset/snowball_fences_fence_off.con"
    end
    
    local result = {}
    local tangent = nil
    
    if segmentType == fences.modeArc and previous then
        tangent = fences.createArc(skipFirstPost, previous, from, to, fence, face, adjust, stepped, offsetZ, invertedModel, result)
    elseif segmentType == fences.modeCurve and previous then
        tangent = fences.createCurve(skipFirstPost, previous, from, to, fence, face, adjust, stepped, offsetZ, invertedModel, result)
    else
        tangent = fences.createLine(skipFirstPost, from, to, fence, face, adjust, stepped, offsetZ, invertedModel, result)
    end
    
    local id = game.interface.buildConstruction(
        constructionId,
        {result = result},
        {1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1})
    
    return id, tangent or from
end

return fences
